// CATALOG -- makes a one-page HTML catalog of your pictures.
// (c) 2004-2005 Lucian Wischik, lu@wischik.com
// Anyone can use this code anyway they wish, apart from claiming ownership or selling it.
//
// The application has a main window and a multiline edit window which totally fills it (used for logs)
// The user drops a directory into the main window. We recursively scan this directory, using
// a recursive function "scan". For each image, we use GDI+ to make a thumbnail of it at least 80x80pix.
// We give thumbnails and filenames to an object called TAccumulator. It crops its thumbnails
// to exactly 80x80pix and adds them into its 12x6 array; once the array is full it saves it to disk
// along with a corresponding <imagemap> in html source code. Everything is triggered by the WM_DROP
// message: this is where we instantiate TAccumulator, write html header+footer, and invoke "scan".
//


#include <windows.h>
#ifndef max          // min/max aren't defined for Borland compilers
#define max(a,b) (((a)>(b))?(a):(b))
#endif
#ifndef min
#define min(a,b) (((a) < (b)) ? (a) : (b))
#endif
#include <gdiplus.h>
#include <shlobj.h>
#include <stdio.h>
#include <list>
#include <string>
#include <fstream>
using namespace std;
using namespace Gdiplus;


HINSTANCE hInstance;
HWND hwnd,hed;          // Main window (initialized in WinMain), and edit control (initialized in WM_CREATE)
bool processing=false;  // 'True' during processing. (during processing, we use PeekMessages periodically)
bool abort_p=false;     // if user clicks WM_CLOSE during processing, we just set this flag rather than closing
int width=12, height=6; // each large jpeg contains width*height thumbnails
CLSID encoderClsid; EncoderParameters encoderParameters; ULONG quality=40; // used for GDI+ jpeg stuff (initalized in WinMain, used when saving jpegs)
wstring StringInt(int i) {wchar_t c[64]; wsprintf(c,L"%ld",i); return wstring(c);}

wstring GetLastErrorString()
{ LPVOID lpMsgBuf; FormatMessage(FORMAT_MESSAGE_ALLOCATE_BUFFER | FORMAT_MESSAGE_FROM_SYSTEM,NULL,
    GetLastError(),MAKELANGID(LANG_NEUTRAL, SUBLANG_DEFAULT), (LPTSTR) &lpMsgBuf, 0, NULL);
  wstring s((wchar_t*)lpMsgBuf); LocalFree( lpMsgBuf ); return s;
}


// print -- prints a log in the edit window "hed"
void print(const wstring msg)
{ wstring t=msg+L"\r\n";
  unsigned int len = GetWindowTextLength(hed);
  if (len>16000) SetWindowText(hed,L"");
  int i=GetWindowTextLength(hed);
  SendMessage(hed,EM_SETSEL,(WPARAM)i,(LPARAM)i);
  SendMessage(hed,EM_REPLACESEL,FALSE,(LPARAM)t.c_str());
}

// during processing, we call PumpMessages periodically to ensure responsiveness
void PumpMessages()
{ MSG msg; for (;;)
  { BOOL res=PeekMessage(&msg,0,0,0,PM_REMOVE);
    if (!res||msg.message==WM_QUIT) return;
    TranslateMessage(&msg); DispatchMessage(&msg);
  }
}



  

// TAccumulator -- user invokes "acc" to accumulate thumbnails. Once there are enough
// of them we flush them all to disk in a big jpeg, along with an html imagemap directive,
// into the directory and html-file specified in the constructor.
//
struct TAccumulator
{ // we will be writing to disk in these places:
  wofstream *html; wstring thumbdir; wstring thumbrdir; int thumbindex;
  // we are building up this object:
  Image *img; int x, y; wstring imgmap;
  //
  TAccumulator(wofstream *_html, wstring _thumbdir, wstring _thumbrdir) : html(_html), thumbdir(_thumbdir), thumbrdir(_thumbrdir), img(0), thumbindex(0) {}

  // acc - Call this to accumulate a thumbnail to an imagemap. If this new thumbnail
  // fills the imagemap size, then flush() is called automatically.
  // Arguments: i2 is a thumbnail image of at least 80x80; we will crop it to 80x80
  // irn is the relative name of the image for href, like "subdir\pic.jpg"
  // alt is the "alt" text when the user hovers over the thumbnail, like "pic.jpg"
  void acc(Image *i2, const wstring irn, const wstring alt)
  { // if we've just started a fresh accumulator-image, then clear it first.
    if (img==0) {img=new Bitmap(80*width,80*height,PixelFormat32bppRGB); x=0; y=0;}
    Graphics *g = new Graphics(img);
    if (x==0 && y==0) 
    { SolidBrush blackBrush(Color(0,0,0));
      g->FillRectangle(&blackBrush,0,0,80*width,80*height);
    }
    // crop and place the thumbnail
    int iw=i2->GetWidth(), ih=i2->GetHeight();
    int ox=(iw-80)/2, oy=(ih-80)/2;
    RectF src((float)ox,(float)oy,80.0,80.0);
    RectF dst((float)(x*80),(float)(y*80),80.0,80.0);
    g->DrawImage(i2,dst,(float)ox,(float)oy,(float)80,(float)80,UnitPixel);
    delete g;
    // add html source code for the imagemap
    wstring wirn = CleanUnicode(irn);
    wstring walt = CleanUnicode(alt);
    imgmap += L"<area shape=rect coords=\""+StringInt(x*80)+L","+StringInt(y*80)+L","+StringInt(x*80+80)+L","+StringInt(y*80+80)+L"\" href=\""+wirn+L"\" alt=\""+walt+L"\"/>\n";
    x++; if (x==width) {x=0; y++;}
    if (y==height) flush();
  }
  wstring CleanUnicode(const wstring t)
  { wstring res;
    unsigned int len=t.length(); for (unsigned int i=0; i<len; i++)
    { if (t[i]=='&') {res+=L"&amp;"; continue;}
      if (t[i]>0 && t[i]<256) {res+=t[i]; continue;}
      wchar_t buf[32]; wsprintf(buf,L"&#%lu;",t[i]); res+=wstring(buf);
    }
    return res;
  }

  // flush: called automatically when acc() exceeds a size. Also can be called by user.
  // It writes the current accumulated thumbnails to a large jpeg file, using gdi+ jpeg-encoder
  // that was initailized in WinMain, and it writes the corresponding imagemap to the html file,
  // and resets the accumulator-image so that next acc() will start a fresh one.
  void flush()
  { if (img==0) return;
    wstring fn = StringInt(thumbindex)+L".jpg";
    wstring tfn = thumbdir+L"\\"+fn;
    wstring trn = thumbrdir+L"\\"+fn;
    img->Save(tfn.c_str(), &encoderClsid, &encoderParameters);
    *html << L"<img src=\"" << thumbrdir << L"/" << StringInt(thumbindex) << L".jpg\" width=" << StringInt(width*80) << L" height=" << StringInt(height*80) << L" usemap=\"#map" << StringInt(thumbindex) << L"\"><br>\n";
    *html << L"<map name=\"map" << StringInt(thumbindex) << L"\">\n";
    *html << imgmap;
    *html << L"</map>\n";
    imgmap=L""; delete img; img=0; x=0; y=0;
    thumbindex++;
  }
};




// scan -- this function recursively scans a directory looking for images.
// When it finds an image it generates a thumbnail of at least 80x80
// and adds it to the TAccumulator. The function calls PumpMessages periodically
// to allow for user responsiveness, and if ever the global flag "abort_p" is
// set then it aborts.
void scan(const wstring base, const wstring subdir, TAccumulator &acc)
{ wstring subprefix=subdir; if (subprefix.length()>0) subprefix+=L"\\";
  wstring dir=base+L"\\"+subprefix;
  wstring s = dir+L"*";
  WIN32_FIND_DATA fdat;
  // 1. look for files
  HANDLE hfind = FindFirstFile(s.c_str(),&fdat);
  for (BOOL gotmore=(hfind!=INVALID_HANDLE_VALUE); gotmore && !abort_p; gotmore=FindNextFile(hfind,&fdat))
  { if (fdat.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY) continue;
    // get the extension
    const wchar_t *c=fdat.cFileName, *ext=0;
    while (*c!=0) {if (*c=='.' || *c=='\\' || *c=='/' || *c==':') ext=c; c++;}
    if (*ext!='.') ext=L"";
    bool isimg = (_wcsicmp(ext,L".jpg")==0) | (_wcsicmp(ext,L".jpeg")==0) | (_wcsicmp(ext,L".bmp")==0)
                |(_wcsicmp(ext,L".tif")==0) | (_wcsicmp(ext,L".tiff")==0) | (_wcsicmp(ext,L".gif")==0);
    if (!isimg) continue;
    wstring srn = subprefix+fdat.cFileName;
    wstring sfn = dir+fdat.cFileName;
    wstring alt = fdat.cFileName;
    // save the file as a thumbnail
    Image *image=new Image(sfn.c_str());
    double w=image->GetWidth(), h=image->GetHeight();
    double scalew=80.0/w, scaleh=80.0/h;
    double scale=scalew; if (scaleh>scale) scale=scaleh;
    int sw=(int)(w*scale+0.5), sh=(int)(h*scale+0.5);
    if (sw>240) sw=240; if (sh>240) sh=240;
    Image *i2 = image->GetThumbnailImage(sw,sh,0,0);
    acc.acc(i2, srn, alt);
    delete i2;
    delete image;
    print(sfn);
    //
    PumpMessages();
  }
  if (hfind!=INVALID_HANDLE_VALUE) FindClose(hfind);
  // 2. look for directories
  hfind = FindFirstFile(s.c_str(),&fdat);
  for (BOOL gotmore=(hfind!=INVALID_HANDLE_VALUE); gotmore && !abort_p; gotmore=FindNextFile(hfind,&fdat))
  { if (!(fdat.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY)) continue;
    if (fdat.cFileName==wstring(L".") || fdat.cFileName==wstring(L"..") || fdat.cFileName==wstring(L"index")) continue;
    wstring srn = subprefix+fdat.cFileName;
    wstring sfn = dir+fdat.cFileName;
    wstring alt = fdat.cFileName;
    //
    Bitmap bmp(80,80,PixelFormat32bppRGB);
    Graphics graphics(&bmp);
    SolidBrush blackBrush(Color(0,0,0));
    SolidBrush whiteBrush(Color(255,255,255));
    FontFamily fontFamily(L"Arial");
    Font font(&fontFamily,12,FontStyleRegular,UnitPixel);
    RectF bounds(0.0f,0.0f,80.0f,80.0f);
    StringFormat format;
    format.SetAlignment(StringAlignmentCenter);
    format.SetLineAlignment(StringAlignmentCenter);
    graphics.FillRectangle(&blackBrush,0,0,80,80);
    graphics.DrawString(alt.c_str(),-1,&font,bounds,&format,&whiteBrush);

    acc.acc(&bmp, srn, alt);
    PumpMessages();
    //
    wstring newsubdir=subdir; if (subdir.length()>0) newsubdir+=L"\\"; newsubdir+=fdat.cFileName;
    scan(base, newsubdir, acc);
  }
  if (hfind!=INVALID_HANDLE_VALUE) FindClose(hfind);
}


LRESULT CALLBACK PlainWndProc(HWND hwnd, UINT msg, WPARAM wParam, LPARAM lParam)
{ switch (msg)
  { case WM_CREATE:
    { hed = CreateWindow(L"EDIT",L"",WS_CHILD|WS_VISIBLE|WS_HSCROLL|WS_VSCROLL|ES_LEFT|ES_MULTILINE,
                    0,0,100,100,hwnd,NULL,hInstance,0);
      print(L"Drop a picture-folder here...");
      print(L"I will make a catalogue in it,");
      print(L"of its contents.");
      print(L"(overwriting index.html and index\\*.jpg)");
      DragAcceptFiles(hwnd,TRUE);
    } break;
    case WM_SIZE:
    { RECT rc; GetClientRect(hwnd,&rc);
      MoveWindow(hed,0,0,rc.right,rc.bottom,TRUE);
    } break;
    case WM_PAINT:
    { PAINTSTRUCT ps; BeginPaint(hwnd,&ps);
      EndPaint(hwnd,&ps);
    } break;
    case WM_DROPFILES:
    { HDROP hdrop = (HDROP)wParam;
      int num = DragQueryFile(hdrop,0xFFFFFFFF,NULL,0);
      if (num>1)
      { DragFinish(hdrop);
        MessageBox(hwnd,L"Please drag just a single folder",L"Catalogue",MB_OK);
        break;
      }
      int len = DragQueryFile(hdrop,0,0,0);
      wchar_t *c=new wchar_t[len+1]; DragQueryFile(hdrop,0,c,len+1);
      wstring dir(c); delete[] c; print(dir);
      abort_p=false;
      processing=true;
      //
      wstring indexdir=wstring(dir)+L"\\index"; CreateDirectory(indexdir.c_str(),0); 
      char fn[MAX_PATH]; WideCharToMultiByte(CP_ACP,0,dir.c_str(),-1,fn,MAX_PATH,0,0); strcat(fn,"\\index.html");
      wofstream html(fn,ios_base::out);
      html << L"<html>\n<head>\n<title>Pictures</title>\n<style type=\"text/css\">\n body {background-color: black; color: white;}\n img {border: 0;}\n</style>\n<body>\n\n";
      TAccumulator acc(&html,indexdir,L"index");
      scan(dir,L"",acc);
      acc.flush();
      html << L"\n</body>\n</html>\n";
      html.close();
      //
      processing=false;
      DragFinish(hdrop);
      print(L"done.");
      print(L"");
      print(L"Drop more picture-folders here if you want...");
      if (abort_p) PostMessage(hwnd,WM_CLOSE,0,0);
    } break;
    case WM_CLOSE:
    { if (processing) {abort_p=true; return 0;}
      else {DestroyWindow(hwnd); return 0;}
    } 
    case WM_DESTROY:
    { DragAcceptFiles(hwnd,FALSE);
      PostQuitMessage(0);
    } break;
  }
  return DefWindowProc(hwnd, msg, wParam, lParam);
}





int WINAPI WinMain(HINSTANCE h,HINSTANCE,LPSTR,int)
{ hInstance=h;
  abort_p=false;
  processing=false;
  //
  GdiplusStartupInput gdiplusStartupInput; ULONG_PTR gdiplusToken;
  GdiplusStartup(&gdiplusToken, &gdiplusStartupInput, NULL);
  // get clsid of the jpeg-encoder
  bool foundencoder=false; UINT num=0, size=0; GetImageEncodersSize(&num, &size); 
  if (size!=0)
  { ImageCodecInfo* pImageCodecInfo = (ImageCodecInfo*)(malloc(size));
    GetImageEncoders(num, size, pImageCodecInfo);
    for (unsigned int j=0; j<num; j++)
    { if (wcscmp(pImageCodecInfo[j].MimeType,L"image/jpeg")==0)
      { encoderClsid=pImageCodecInfo[j].Clsid; foundencoder=true;
      }    
    }
    free(pImageCodecInfo);
  }
  if (!foundencoder) {MessageBox(NULL,L"Failed to find GDI+ jpeg",L"Error",MB_OK); GdiplusShutdown(gdiplusToken); return 0;}
  // and configure the jpeg-encoder
  encoderParameters.Count = 1;
  encoderParameters.Parameter[0].Guid = EncoderQuality;
  encoderParameters.Parameter[0].Type = EncoderParameterValueTypeLong;
  encoderParameters.Parameter[0].NumberOfValues = 1;
  encoderParameters.Parameter[0].Value = &quality;
  //
  WNDCLASSEX wcex; ZeroMemory(&wcex,sizeof(wcex)); wcex.cbSize = sizeof(WNDCLASSEX);
  BOOL res=GetClassInfoEx(hInstance,L"PlainClass",&wcex);
  if (!res)
  { wcex.style = CS_HREDRAW | CS_VREDRAW;
    wcex.lpfnWndProc = (WNDPROC)PlainWndProc;
    wcex.cbClsExtra = 0;
    wcex.cbWndExtra = 0;
    wcex.hInstance = hInstance;
    wcex.hIcon = NULL;
    wcex.hCursor = LoadCursor(NULL, IDC_ARROW);
    wcex.hbrBackground = (HBRUSH)GetStockObject(WHITE_BRUSH);
    wcex.lpszMenuName	= NULL;
    wcex.lpszClassName = L"PlainClass";
    wcex.hIconSm = NULL;
    ATOM res=RegisterClassEx(&wcex);
    if (res==0) {MessageBox(NULL,L"Failed to register class",L"Error",MB_OK); GdiplusShutdown(gdiplusToken); return 0;}
  }
  //
  hwnd = CreateWindowEx(0,L"PlainClass", L"Catalogue Pictures", WS_OVERLAPPEDWINDOW,
      CW_USEDEFAULT, CW_USEDEFAULT, 400, 400, NULL, NULL, hInstance, NULL);
  if (hwnd==NULL) {MessageBox(NULL,L"Failed to create window",L"Error",MB_OK); GdiplusShutdown(gdiplusToken); return 0;}
  ShowWindow(hwnd,SW_SHOW);
  //
  MSG msg;
  while (GetMessage(&msg, NULL, 0, 0))
  { TranslateMessage(&msg);
	  DispatchMessage(&msg);
  }
  //
  GdiplusShutdown(gdiplusToken);
  return msg.wParam;
}

